/* 
 *  CodeSelectionForm.java
 * 
 *  Copyright 2016 Avaya Inc. All Rights Reserved.
 * 
 *  Usage of this source is bound to the terms described
 *  in AvayaLicenseSDK.rtf.
 * 
 *  Avaya - Confidential & Proprietary. Use pursuant to your signed agreement
 *  or Avaya Policy
 * 
 */
package com.avaya.ccs.javafxrefclient;

import com.avaya.ccs.api.CodeListI;
import com.avaya.ccs.api.ContactCenterCodeI;
import javafx.beans.value.ChangeListener;
import javafx.beans.value.ObservableValue;
import javafx.collections.FXCollections;
import javafx.collections.ObservableList;
import javafx.event.ActionEvent;
import javafx.scene.Scene;
import javafx.scene.control.Button;
import javafx.scene.control.ListCell;
import javafx.scene.control.ListView;
import javafx.scene.layout.BorderPane;
import javafx.scene.layout.HBox;
import javafx.stage.Stage;
import javafx.util.Callback;

public class CodeSelectionForm {

    private final Stage codeSelection = new Stage();
    private static final Logger LOG = Logger.getLogger(CodeSelectionForm.class);
    private final UserExecutor user;
    private final InteractionExecutor interactionExe; 

    public CodeSelectionForm(UserExecutor user, InteractionExecutor interactionExe, String css) {
        this.user = user;
        this.interactionExe = interactionExe;
        BorderPane codeSelectionBorder = new BorderPane();
        Scene codeSelectionScene = new Scene(codeSelectionBorder, 300, 400);
        codeSelectionScene.getStylesheets().add(css);
        codeSelection.setScene(codeSelectionScene);
        codeSelection.titleProperty().set("Code Selection");

    }

    public void show() {
        codeSelection.show();
    }

    public void updateCodeSectionForm(CodeListI codes, String currentSelectionID) {
        String methodName = "updateCodeSectionForm() ";
        LOG.trace(methodName + "ListType:" + codes.getCodeType() + " size " + codes.getCodes().size());
        BorderPane bp = (BorderPane) codeSelection.sceneProperty().get().getRoot();

        String title = "Unknown Code Type";
        switch (codes.getCodeType()) {
            case NRRC:
                title = "Select Not Ready Reason Code";
                break;
            case ACW:
                title = "Select After Call Work Code";
                break;
            case Activity:
                title = "Select Activity Code";
                break;
            case Disposition:
                title = "Select Disposition Code";
                break;
            default:
            	break;
        }
        codeSelection.setTitle(title);

        ListView<ContactCenterCodeI> list = new ListView<>();
        ObservableList<ContactCenterCodeI> items = FXCollections.observableArrayList(
                codes.getCodes());
        list.setItems(items);
        //We want the list to display the name and code only
        list.setCellFactory(new Callback<ListView<ContactCenterCodeI>, ListCell<ContactCenterCodeI>>() {
            @Override
            public ListCell<ContactCenterCodeI> call(ListView<ContactCenterCodeI> list) {
                return new UserDataGUI.CodeCell();
            }
        }
        );

        //Only enable the ok button when a selection is made
        Button okButton = new Button("OK");
        okButton.setDisable(true);

        list.getSelectionModel().selectedItemProperty().addListener(new ChangeListener<ContactCenterCodeI>() {
            @Override
            public void changed(ObservableValue<? extends ContactCenterCodeI> observable, ContactCenterCodeI oldValue, ContactCenterCodeI newValue) {
                if (newValue != null) {
                    okButton.setDisable(false);
                }
            }
        }
        );

        bp.setCenter(list);

        okButton.getStyleClass().add("interactionButton");
        okButton.setOnAction((ActionEvent t) -> {
            ContactCenterCodeI selectedCode = list.getSelectionModel().getSelectedItem();
            if (selectedCode == null) {
                LOG.info(methodName + "OKButton(): no value selected");
                return;
            }

            switch (codes.getCodeType()) {
                case NRRC:
                    LOG.info(methodName + "OKButton():Not Ready Reason Code:" + selectedCode);
                    this.user.notReady(currentSelectionID, selectedCode.getNumber().toString());
                    break;
                case ACW:
                    LOG.info(methodName + "OKButton():After Call Work Code:" + selectedCode);
                    this.user.ACW(currentSelectionID, selectedCode.getNumber().toString());
                    break;
                case Activity:
                    LOG.info(methodName + "OKButton():Activity Code:" + selectedCode);
                    this.interactionExe.setActivityCode(currentSelectionID, selectedCode.getNumber().toString());
                    break;
                case Disposition:
                    LOG.info(methodName + "OKButton():Disposition Code:" + selectedCode);
                    this.interactionExe.wrapUPWithCode(currentSelectionID, selectedCode);
                    break;
                default:
                	break;
            }
            codeSelection.close();
        });

        Button cancelButton = new Button("Cancel");
        cancelButton.getStyleClass().add("interactionButton");
        cancelButton.setOnAction((ActionEvent t) -> {
            LOG.info("CancelButton()");
            codeSelection.close();
        });

        //Need to be able to clear an activity code, but it is not possible to clear the other codes
        Button clearButton = new Button("Clear");

        switch (codes.getCodeType()) {
            case NRRC:
                clearButton.setDisable(true);
                break;
            case ACW:
                clearButton.setDisable(true);
                break;
            case Activity:
                clearButton.setDisable(false);
                break;
            case Disposition:
            case Unknown:
                clearButton.setDisable(true);
                break;
        }

        clearButton.getStyleClass().add("interactionButton");
        clearButton.setOnAction((ActionEvent t) -> {
            LOG.info(methodName + "ClearButton(): only applicable to activity codes");
            this.interactionExe.setActivityCode(currentSelectionID, null);
            codeSelection.close();
        });

        HBox hbuttonbox = new HBox();
        hbuttonbox.getChildren().addAll(okButton, cancelButton, clearButton);
        bp.setBottom(hbuttonbox);
    }
}
